/////////////////////////////////////////////////////////////////////////
//
//  Institute for Software Science, University of Vienna, 2004
//
// Copyright in this software belongs to Institute for Software Science, 
// University of Vienna, Nordbergstrasse 15/C/3, 1090 Vienna, Austria
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//	Created By :		Gerhard Engelbrecht
//	Created Date :		2004/04/20
//	Created for Project:	GEMSS
//
////////////////////////////////////////////////////////////////////////
//
// Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//	Last commit info:	$Author: gerry $
//					$Date: 2004/12/13 11:12:09 $
//					$Revision: 1.7 $
//
/////////////////////////////////////////////////////////////////////////

package at.ac.univie.iss.service;
 
import java.rmi.RemoteException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.activation.DataHandler;

import at.ac.univie.iss.service.app.AppHandler;
import at.ac.univie.iss.service.qos.QoSHandler;
import at.ac.univie.iss.service.recovery.RecoveryHandler;


/**
*	The <code>ApplicationExcecuter</code> class provides all service methods
*	that are exposed as WSDL operations in a standard web service hosting 
*	environment (e.g. Jakarta Tomcat). An instance of this class can be seen
*	as entry point to a web service (c.f. remote class in Java RMI).
*	<p>
*	The <code>ApplicationExcecuter</code> class provides all methods a client
*	may use via SOAP. This includes the following operations:
*	<ul>
*	<li>upload input data</li>
*	<li>start native application</li>
*	<li>download output data</li>
*	<li>status-queries</li>
*	<li>kill a native application</li>
*	<li>QoS negotiation (if supported)</li>
*	<li>Recovery operations (if supported)</li>
*	</ul>
*
*	@author Gerhard Engelbrecht
*/
final public class ApplicationExecuter implements ApplicationService, QoS, Recovery {

    private static Logger logger = null;
    
    private AppHandler appHandler = null;
    
    private QoSHandler qosHandler = null;
    
    private RecoveryHandler recoveryHandler = null;

    private boolean qosSupported = false;
    private boolean recoverySupported = false;
	

	/**
	*	Creates a new <code>ApplicationExecuter</code> instance. This standard
	*	constructor is needed for each web application and it is executed on
	*	the first request (method invocation) of a client. 
	*	<p>
	*	Internally this constructor tries to create appropriate handlers for
	*	basic application handling (<code>AppHandler</code>) and for QoS 
	*	support (<code>QoSHandler</code>). If no QoS is supported the 
	*	invocation of a QoS method (e.g. <code>requestQoSDescirptor</code>)
	*	throws an Exception.
	*/
	public ApplicationExecuter() throws RemoteException {

		try {
	        logger = Logger.getLogger("logging.appex");
	        
	        logger.log(Level.INFO, "Started.\n");

			// Initialize basic application handler

			HandlerFactory appHandlerFactory = HandlerFactory.getInstance("at.ac.univie.iss.service.app.AppHandler");
			appHandler = (AppHandler)appHandlerFactory.getHandler();
	
			// Trying to initialize QoS handler

			try {
				HandlerFactory qosHandlerFactory = HandlerFactory.getInstance("at.ac.univie.iss.service.qos.QoSHandler");
				qosHandler = (QoSHandler)qosHandlerFactory.getHandler();
				qosHandler.setAppHandler(appHandler);
				qosSupported = qosHandler.isQoSEnabled();
			}
			catch (ClassNotFoundException cnfException) {
				// QoSHandlerFactory not found --> i.e. no Quality of Service supported!
		        logger.log(Level.INFO, "This service does not support Quality of Service methods.\n");
				logger.log(Level.FINEST, "An error occurs while trying to load QoSHandler implementation.\n", cnfException);
			}
			catch (Exception qosException) {
				logger.log(Level.WARNING, "An error occurs while trying to initialize the QoS support (see details below).\n", qosException);
			}

			// Trying to initialize Recovery handler

			try {
				HandlerFactory recoveryHandlerFactory = HandlerFactory.getInstance("at.ac.univie.iss.service.qos.RecoveryHandler");
				recoveryHandler = (RecoveryHandler)recoveryHandlerFactory.getHandler();
				recoverySupported = true;
			}
			catch (ClassNotFoundException cnfException) {
				// RecoveryHandlerFactory not found --> i.e. no Recovery supported!
		        logger.log(Level.INFO, "This service does not support Recovery methods.\n");
				logger.log(Level.FINEST, "An error occurs while trying to load RecoveryHandler implementation.\n", cnfException);
			}
			catch (Exception recoveryException) {
				logger.log(Level.WARNING, "An error occurs while trying to initialize the Recovery support (see details below).\n", recoveryException);
			}
		}
		catch (Exception exception) {
			
	        logger.log(Level.SEVERE, "An unrecoverable error occured!\n", exception);
			throw (new RemoteException("ApplicationExecuter failed!", exception));
		}
    }

    public String requestQoSDescriptor(String qoSDescriptor, String requestDescriptor) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			return qosHandler.requestQoSDescriptor(qoSDescriptor, requestDescriptor);
		}
		throw (new RemoteException("This service does NOT provide QoS support!"));
    }
	
    public String confirmQoSDescriptor(String qoSDescriptor) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
	        return qosHandler.confirmQoSDescriptor(qoSDescriptor);
		}
		throw (new RemoteException("This service does NOT provide QoS support!"));
    }
	
    public void cancelQoSDescriptor(String qoSDescriptor) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
	        qosHandler.cancelQoSDescriptor(qoSDescriptor);
	        return;
		}
		throw (new RemoteException("This service does NOT provide QoS support!"));
    }
	
    public void upload(String cid, byte[] data) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			qosHandler.upload(cid, data);
		}
		else {
	        appHandler.upload(cid, data);
	    }
    }
	
	public byte[] download(String cid, String fileName) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			return qosHandler.download(cid, fileName);
		}
		else {
	        return appHandler.download(cid, fileName);
	    }
	}
	
    public void uploadAttachment(String cid, DataHandler dataHandler) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");
        
		if (qosSupported) {
			qosHandler.uploadAttachment(cid, dataHandler);
		}
		else {
	        appHandler.uploadAttachment(cid, dataHandler);
	    }
    }
	
	public DataHandler downloadAttachment(String cid, String fileName) throws RemoteException {
		
        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			return qosHandler.downloadAttachment(cid, fileName);
		}
		else {
	        return appHandler.downloadAttachment(cid, fileName);
	    }
	}

    public void uploadString(String cid, String data) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			qosHandler.uploadString(cid, data);
		}
		else {
	        appHandler.uploadString(cid, data);
	    }
    }
	
	public String downloadString(String cid, String fileName) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			return qosHandler.downloadString(cid, fileName);
		}
		else {
	        return appHandler.downloadString(cid, fileName);
	    }
	}

    public void start(String cid) throws RemoteException {
        
        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			qosHandler.start(cid);
		}
		else {
	        appHandler.start(cid);
	    }
    }
	
	public void kill(String cid) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");
	
		if (qosSupported) {
			qosHandler.kill(cid);
		}
		else {
	        appHandler.kill(cid);
	    }
	}
	
    public byte[] getStatus(String cid) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");
        
		if (qosSupported) {
			return qosHandler.getStatus(cid);
		}
		else {
	        return appHandler.getStatus(cid);
	    }
    }

    public String getStatusAsString(String cid) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			return qosHandler.getStatusAsString(cid);
		}
		else {
	        return appHandler.getStatusAsString(cid);
	    }
    }

    public void acknowledgeResults(String cid) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (qosSupported) {
			qosHandler.acknowledgeResults(cid);	
		}
		else {
	        appHandler.acknowledgeResults(cid);	
	    }
    }

    public String getCId() throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");
        
        return appHandler.getCId();
    }

	public void restart(String cid) throws RemoteException {

        logger.log(Level.FINEST, "Started.\n");

		if (recoverySupported) {
			recoveryHandler.restart(cid);
		}
	}
	
	public DataHandler checkpointDownloadAttachment(String cid, String fileName) throws RemoteException {
		
        logger.log(Level.FINEST, "Started.\n");

		if (recoverySupported) {
			return recoveryHandler.checkpointDownloadAttachment(cid, fileName);
		}
		return null;
	}

	public void checkpointUploadAttachment(String cid, DataHandler dataHandler) throws RemoteException {
		
        logger.log(Level.FINEST, "Started.\n");

		if (recoverySupported) {
			recoveryHandler.checkpointUploadAttachment(cid, dataHandler);
		}
	}

}
