/////////////////////////////////////////////////////////////////////////
//
//  Institute for Software Science, University of Vienna, 2004
//
// Copyright in this software belongs to Institute for Software Science, 
// University of Vienna, Nordbergstrasse 15/C/3, 1090 Vienna, Austria
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//	Created By :		Gerhard Engelbrecht
//	Created Date :		2004/04/20
//	Created for Project:	GEMSS
//
////////////////////////////////////////////////////////////////////////
//
// Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//	Last commit info:	$Author: gerry $
//					$Date: 2004/05/30 21:39:46 $
//					$Revision: 1.5 $
//
/////////////////////////////////////////////////////////////////////////

package at.ac.univie.iss.service;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
*	Abstract Factory for creating concrete handler instances.<p>
*	The generic application service utilizes the Abstract Factory software
*	design pattern to create handlers for basic application functionality
*	(upload, start etc. in the <code>AppHandler</code> class) and
*	Quality of Service Support (requestQoSDescriptor, etc. in the <code>
*	QoSHandler</code>). This <code>HandlerFactory</code> class is utilized
*	as abstract factory class for this purpose. Concrete extensions of this
*	class are:
*	<ul>
*	<li><code>AppHandlerFactory</code></li>
*	<li><code>QoSHandlerFactory</code></li>
*	</ul>
*
*	@author	Gerhard Engelbrecht
*
*	@see	at.ac.univie.iss.service.app.AppHandlerFactory
*	@see	at.ac.univie.iss.service.qos.QoSHandlerFactory
*/
abstract public class HandlerFactory {

    /**
    *	Default logger for all handler factories (uses "appex" as logger 
    *	namespace).
    */
	protected static Logger logger = Logger.getLogger("logging.appex");

	protected HandlerFactory() {
	}

    /**
    *	Returns a concrete <code>HandlerFactory</code> for the the given
    *	class name (<code>AppHandler</code> or <code>QoSHandler</code>).
    *	
    *	@return		new <code>HandlerFactory</code> instance.
    *
    *	@throws		Exception if the concrete <code>HandlerFactory</code>
    *				class could not be initialized.
    */
	final static public HandlerFactory getInstance(String handlerName) throws Exception {

    	logger.log(Level.FINER, "Started.\n");

		Class cl = Class.forName("at.ac.univie.iss.service.HandlerFactory");
		
    	logger.log(Level.FINEST, "Class object for HandlerFactory class created.\n");

    	ClassLoader factoryLoader = cl.getClassLoader();
    	
    	logger.log(Level.FINEST, "Class loader of HandlerFactory class created.\n");
	    
    	Class factoryClass = factoryLoader.loadClass(handlerName + "Factory");
	    	
	    logger.log(Level.FINEST, "Factory class object (" + factoryClass + ") created.\n");

		Object obj = factoryClass.newInstance();
		
	    logger.log(Level.FINEST, "Factory instance (" + obj + ") created.\n");

    	return ((HandlerFactory) obj);
	}
	
    /**
    *	Abstract method that returns a concrete handler.
    *
    *	@throws		Exception if the concrete handler creation failed.
    */
    abstract public Object getHandler() throws Exception;
}
