/////////////////////////////////////////////////////////////////////////
//
//  Institute for Software Science, University of Vienna, 2004
//
// Copyright in this software belongs to Institute for Software Science, 
// University of Vienna, Nordbergstrasse 15/C/3, 1090 Vienna, Austria
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//	Created By :		Gerhard Engelbrecht
//	Created Date :		2004/04/20
//	Created for Project:	GEMSS
//
////////////////////////////////////////////////////////////////////////
//
// Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//	Last commit info:	$Author: gerry $
//					$Date: 2004/07/21 14:45:04 $
//					$Revision: 1.1 $
//
/////////////////////////////////////////////////////////////////////////

package at.ac.univie.iss.service.state;

import java.io.Serializable;
import java.util.Vector;

/**
*	The <code>StateDetails</code> class holds state information for one
*	specific client (consumer) - service (provider) relation.<p>
*	Every interaction between a client and the service (i.e. a method
*	invocation) changes this <code>StateDetails</code> object e.g. by 
*	storing a new state in the current state information.
*
*	@author		Gerhard Engelbrecht
*/
final public class StateDetails implements Serializable {

	private String currentState = null;
	private Vector previousStates = null;

    /**
    *	Creates a new and empty <code>StateDetails</code> object.
    */
	public StateDetails() {

		currentState = null;
		previousStates = new Vector();
	}

	/**
	*	Sets the current state of this client-service relation. This method
	*	is usually called at the beginning and at the end of a service
	*	method invocation of a client. For example during the invocation of 
	*	the <code>upload</code> in the application service the state is first
	*	set to UPLOADING and if it has completed successfully the state is 
	*	set to UPLOADED. 
	*
	*	@param	currentState current state
	*/
	public void setState(String currentState) {

		if (this.currentState != null) {	
			this.addState(this.currentState);
		}
		this.currentState = currentState;
	}

	private void addState(String state) {

		previousStates.add(new TSState(state));
	}
	
	/**
	*	Returns the current state of this client-service relation.
	*
	*	@return	current state
	*/
	public String getState() {
		
		return this.currentState;
	}

	/**
	*	Returns all previously set states in a <code>Vector</code> object.
	*
	*	@return	all previously set states
	*/
	public Vector getPreviousStates() {
	
		return this.previousStates;
	}
	
	/**
	*	Returns this <code>StateDetails</code> object as String.
	*
	*	@return	String of this <code>StateDetails</code>
	*/
	public String toString() {
	
		StringBuffer buffer = new StringBuffer();

		buffer.append("[State: " + this.currentState);
		buffer.append(", Previous States: [");
		
		for (int count = 0; count < previousStates.size(); count++) {
			
			if (count > 0) {

				buffer.append(", ");
			}
			buffer.append(previousStates.get(count));
		}
		buffer.append("]]");

		return buffer.toString();
	}
}