/*
 ************************************************************************
 *
 * Copyright (c) 2003-2004, C&C Research Laboratories, NEC Europe Ltd.
 *
 * Copyright in this software belongs to C&C Research Laboratories,
 * Rathausallee 10, 53757 Sankt Augustin, Germany.
 *
 * This software may not be used, sold, licensed, transferred, copied
 * or reproduced in whole or in part in any manner or form or in or
 * on any media by any person other than in accordance with the terms
 * of the Licence Agreement supplied with the software, or otherwise
 * without the prior written consent of the copyright owners.
 *
 * This software is distributed WITHOUT ANY WARRANTY, without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE, except where stated in the Licence Agreement supplied with
 * the software.
 *
 *  Created By :           G.A. Kohring
 *  Created for Project :  GEMSS (IST-2001-37153)
 *
 ************************************************************************
 *
 */

package de.nece.ccrle.docmonitor;


import java.net.URL;
import java.io.File;

/**
 * An unmutable class for holding document attributes, in particular
 * access related attributes. Access related attributes
 * describe the authentication procedure (if any), along with the information 
 * needed for accessing a given url.
 *
 * @author Greg Kohring
 */
public class DocAttributes 
{
    // The URL of the remote document.
    private URL url = null;

    // The local copy of the remote document.
    private File localFile = null;

    // Authentication scheme at remote URL site 
    private String authScheme = null;

    // User name for authentication
    private String user = null;

    // Password for authentication
    private String password = null;

    // The procedure for accessing this document (e.g., "replace" the local 
    // copy or "update" only if the the remote URL is newer)
    private String procedure = null;

    /**
     * Create a new object with the specified parameters.
     * 
     * @param url the <code>URL</code> of the document to be retrieved. 
     * @param procedure a <code>String</code> encoding the procedure
     *      to be used when updating the file.
     * @param authScheme a <code>String</code> containing the name of the
     *      authentication scheme used to protect the document.
     * @param user a <code>String</code> containing the name to supply for
     *      the user parameter in the specified authentication scheme,
     *      or <code>null</code> if no authentication
     *      scheme is being used.
     * @param password a <code>String</code> containing the password to 
     *      supply for the password parameter in the specified 
     *      authentication scheme, or <code>null</code> if no authentication
     *      scheme is being used.
     */
    public DocAttributes( URL url, File localFile, String procedure, 
                          String authScheme, String user, String password ) {
        this.url = url;
        this.localFile = localFile;
        if ( authScheme == null ) {
            this.authScheme = "";
        } else {
            this.authScheme = authScheme;
        }
        this.user = user;
        this.password = password;
        this.procedure = procedure;
    }

    /**
     * Retrieve the URL of the remote document.
     *
     * @return the <code>URL</code> of the document to be retrieved.
     */
    public URL getURL(){return url;}

    /**
     * Retrieve the file for the local copy of the remote document.
     *
     * @return the local <code>File</code> for storing the document.
     */
    public File getLocalFile(){
        return localFile;
    }


    /**
     * Retrieve the authenication scheme.
     *
     * @return a non-<code>null</code> <code>String</code> containg the 
     *      authentication scheme
     *      used to protect the document. If the document is not protected by
     *      an authentication scheme the returned string is empty.
     */
    public String getAuthScheme(){
        return authScheme;
    }

    /**
     * Retrieve the procedure for updating this document.
     *
     * @return a <code>String</code> containing the procedure to use.
     */
    public String getProcedure(){
        return procedure;
    }


    /**
     * Retrieve the user name.
     *
     * @return a <code>String</code> containg the user name
     *      to be supplied as part of the authentication scheme. May 
     *      return <code>null</code> if the 
     *      document is not protected by an authentication scheme.
     */
    protected String getUser(){
        return user;
    }

    /**
     * Retrieve the password corresponding to the user name.
     *
     * @return a <code>String</code> containg the password
     *      to be supplied as part of the authentication scheme. May 
     *      return <code>null</code> if the 
     *      document is not protected by an authentication scheme.
     */
    protected String getPassword(){
        return password;
    }

}
