/*
 ************************************************************************
 *
 * Copyright (c) 2003-2004, C&C Research Laboratories, NEC Europe Ltd.
 *
 * Copyright in this software belongs to C&C Research Laboratories,
 * Rathausallee 10, 53757 Sankt Augustin, Germany.
 *
 * This software may not be used, sold, licensed, transferred, copied
 * or reproduced in whole or in part in any manner or form or in or
 * on any media by any person other than in accordance with the terms
 * of the Licence Agreement supplied with the software, or otherwise
 * without the prior written consent of the copyright owners.
 *
 * This software is distributed WITHOUT ANY WARRANTY, without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE, except where stated in the Licence Agreement supplied with
 * the software.
 *
 *  Created By :           G.A. Kohring
 *  Created for Project :  GEMSS (IST-2001-37153)
 *
 ************************************************************************
 */

package de.nece.ccrle.organizer;

import java.util.Set;
import java.util.Iterator;
import java.io.IOException;

/**
 * A repository is an abstraction of a permanent storage device.  Objects
 * stored in a repository are expected to have a long life-time, i.e., to
 * be reinstantiatable the next time the application is launched.
 *
 * @author Greg Kohring
 */
public interface Repository {

    /**
     * Returns the item corresponding to the specified key.
     * 
     * @param key the key corresponding to the item to be retrieved.
     * @return the item associated with the specified key or 
     *      <code>null</code> if the specified key is not in use.
     */
    public Object getItem(Object key);

    /**
     * Returns an iterator over the items in this repository.
     * 
     * @return an <code>Iterator</code> over the items stored in this
     *      repository.
     */
    public Iterator iterator();

    /**
     * Retrieves the keys known to this repository.
     * 
     * @return an array of <code>Object</code>s which serve as keys for the
     *      items in this repository.
     */
    public Object[] getKeys();

    /**
     * Associates the specified item with the specified key (optional
     * operation). If the specified key is already in use, the item will
     * not be added.
     *
     * @param key the key to be associated with the specified item.
     * @param item the item to be added to the repository.
     * @return <code>true</code> if the repository was changed as a result
     *      of this operation.
     * @throws UnsupportedOperationException if this repository does support
     *      adding additional objects.
     */
    public boolean add( Object key, Object item );

    /**
     * Removes the items associated with the specified key from the repository
     * (optional operation).
     *
     * @param key the key identifying the object to be removed.
     * @return <code>true</code> if the repository was changed as a result
     *      of this operation.
     * @throws UnsupportedOperationException if this repository does support
     *      adding additional objects.
     */
    public boolean remove( Object key ) throws IOException ;

    /**
     * Returns the number of items in the repository.
     * <p>
     * If the repository has more than <code>Integer.MAX_VALUE</code> elements,
     * then the number <code>Integer.MAX_VALUE</code> is returned. 
     * 
     * @return an <code>int</code> containing the number of items in the 
     *      repository.
     */
    public int size();

    /**
     * Determines whether or not the repository is empty.
     * <p>
     * 
     * @return <code>true</code> if the repository is empty, or
     *      <code>false</code> otherwise.
     */
    public boolean isEmpty();

    /**
     * Determines whether or not this repository is equal to the specified
     * object.
     * <p>
     * 
     * @return <code>true</code> if this repository is equal to the specified
     *      repository and false otherwise.
     */
    public boolean equals(Object o);

    /**
     * Retrieves the hash code for this repository.
     * 
     * @return an <code>int</code> containing the hashcode for this
     * repository.
     */
    public int hashCode();
}
