/*
 ************************************************************************
 *
 * Copyright (c) 2003-2004, C&C Research Laboratories, NEC Europe Ltd.
 *
 * Copyright in this software belongs to C&C Research Laboratories,
 * Rathausallee 10, 53757 Sankt Augustin, Germany.
 *
 * This software may not be used, sold, licensed, transferred, copied
 * or reproduced in whole or in part in any manner or form or in or
 * on any media by any person other than in accordance with the terms
 * of the Licence Agreement supplied with the software, or otherwise
 * without the prior written consent of the copyright owners.
 *
 * This software is distributed WITHOUT ANY WARRANTY, without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE, except where stated in the Licence Agreement supplied with
 * the software.
 *
 *  Created By :           G.A. Kohring
 *  Created for Project :  GEMSS (IST-2001-37153)
 *
 ************************************************************************
 */

package de.nece.ccrle.organizer;

import java.io.File;
import java.io.IOException;
import java.util.Collection;

/**
 * A repository for session objects.
 *
 *
 * @author Greg Kohring
 */
public interface SessionRepository {
    /**
     * Retrieves a readable session object from the session repository.
     *
     * @param sessionID a <code>String</code> identifying the session to be
     *      retrieved.
     * @return a <code>Session</code> object for reading from the
     *      specified session
     * @throws IOException if an I/O exception occurs while
     *      opening the specified session for reading.
     */
    public Session getSession( String sessionID ) throws IOException;

    /**
     * Retrieves a list of the sessions contained in this repository.
     *
     * @return an array of <code>String</code>s containing the names of
     *      known sessions.
     */
    public String[] sessions();

    /**
     * Removes the specified string from this repository. Once a session has
     * been deleted it cannot be retrieved.
     *
     * @param sessionID a <code>String</code> specifying the session to be
     *      deleted.
     * @throws IOException if an I/O exception occurs while
     *      deleting the specified session.
     */
    public boolean removeSession( String sessionID ) throws IOException;

    /**
     * Adds the specified session under the specified session id to the
     * repository.
     * <p> 
     * A session is saved a byte stream having the following format:
     * <pre>
     *      sessionID:string
     * </pre>
     * followed by a series of the following:
     * <pre>
     *      maker:byte
     *      key:String
     *      component: Object
     * </pre>
     * The  marker is a single byte used to separate the components.
     * It has the value <code>0xbb</code>
     * Each class is annotated with the following information:
     * <pre>
     *      class loader id: String
     * </pre>
     * A class loader id equal to "default" implies the default class loader.
     *
     * @param sessionID the sessionID to be associated with the specified
     *      session.
     * @param session the session to be added to the repository.
     * @return <code>true</code> if the repository was changed as a result
     *      of this operation.
     */
    boolean addSession(String sessionID, Session session);

    /**
     * Creates a new session object
     *
     * @return a <code>Session</code> object for storing a new session
     * @throws IOException if an I/O exception occurs while
     *      creating the specified session.
     */
    public Session createSession( String sessionID ) throws IOException ;

}
