/*
 ************************************************************************
 *
 * Copyright (c) 2003-2004, C&C Research Laboratories, NEC Europe Ltd.
 *
 * Copyright in this software belongs to C&C Research Laboratories,
 * Rathausallee 10, 53757 Sankt Augustin, Germany.
 *
 * This software may not be used, sold, licensed, transferred, copied
 * or reproduced in whole or in part in any manner or form or in or
 * on any media by any person other than in accordance with the terms
 * of the Licence Agreement supplied with the software, or otherwise
 * without the prior written consent of the copyright owners.
 *
 * This software is distributed WITHOUT ANY WARRANTY, without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE, except where stated in the Licence Agreement supplied with
 * the software.
 *
 *  Created By :           G.A. Kohring
 *  Created for Project :  GEMSS (IST-2001-37153)
 *
 ************************************************************************
 */

package de.nece.ccrle.sandbox;

import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.Provider;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.Collection;
import java.util.List;

/**
 * An interface for managing trust.
 *
 * @author Greg Kohring
 */
public abstract class TrustManager {
    /**
     * Checks whether or not one of the specified certificate chains is 
     * trusted for the purpose of code signing. 
     * <p>
     * The certificate chains are assumed to be arranged in the standard
     * format returned by the <code>getCertificates()</code> method of 
     * {@link java.security.CodeSource}, i.e., the certificates belonging to
     * separate chains are placed consecutively into the array, whereby the
     * first certificate in each chain is the code signing certificate for
     * that chain.
     * If there exists at least one chain trusted for code signing,
     * then this method returns a list containing all valid certificate chains,
     * otherwise it will throw a
     * <code>CertificateException</code>.
     *
     * @param chains an <code>array</code> of certificates, consisting of one
     *      or more certificate chains to be checked.  The first certificate
     *      in each chain should be the code signing certificate.
     * @return a <code>List</code> containing one 
     *      <code>CertPathBuilderResult</code> for each valid signing chain
     *      found in the specified array.
     * @throws CertificateException if no valid chain is found.
     */
    public abstract List checkCodeSigningTrusted( Certificate[] chains ) 
                                        throws CertificateException;

    /**
     * Checks whether or not the specified certificate chain is trusted based
     * upon the trusted certificates known to this trust manager.
     *
     * @param chain an <code>array</code> of certificates to check.
     * @throws CertificateException if the chain cannot be verified.
     */
    public abstract List checkTrusted( Certificate[] chain )
                                throws CertificateException;

    /**
     * Instruct the TrustManager to use the specified security provider.
     * If no security provider has been set, then the default provider is
     * used.
     *
     * @param securityProvider the security <code>Provider</code> to be used.
     */
    abstract void setSecurityProvider( Provider securityProvider );

    /**
     * Retrieves the current security provider.
     *
     * @return the security <code>Provider</code> currently in use or 
     *      <code>null</code> if no <code>Provider</code> has been set.
     */
    abstract Provider getSecurityProvider();

    /**
     * Checks whether the specified certificate is known under the
     * specified alias.
     *
     * @param cert the <code>Certificate</code> to be checked.
     * @param alias an alias known to the <code>TrustManager</code>
     * @return <code>true</code> if the specified <em>certificate</em> is known
     *      under the specified <em>alias</em> and <code>false</code> otherwise.
     */
    abstract boolean knownAs( Certificate cert, String alias );

    /**
     * Add the certificates from the specified keystore to the list of
     * know
     *
     * @param trustStore a <code>KeyStore</code> containing trusted
     *      <code>Certificates</code>.
     * @param keyStoreType a <code>String</code> identifying the
     *      the type of keystore.
     */
    abstract void addTrustedCerts( KeyStore trustStore ) 
                        throws KeyStoreException;

}
