/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package eu.gemss.components.negotiation;

import eu.gemss.components.proxies.GEMSSProxy;
import java.util.Vector;
import java.util.Hashtable;
import java.io.Serializable;

/**
 * This interface defines the functions supported by the proposal history object, used by the QoSNegotiation component.
 */
public interface ProposalHistory extends Serializable {

      /**
       * Add a party to the auction history. Parties will participate in the auction, and be sent the relevant auction
       * messages etc.
       * @param proxy GEMSSProxy bound to a service provider
       */
      public void addParty( GEMSSProxy proxy ) throws Exception;

      /**
       * Add a WSLA object to the auction history. Parties will each require a WSLA before the auction can be started.
       * Exceptions are passed on through.
       * @param proxy GEMSSProxy bound to a service provider
       * @param strWSLA WSLA object
       */
      public void addWSLATemplate( GEMSSProxy proxy, String strWSLA ) throws Exception;

      /**
       * Set the active call for proposals for this proxy. The call is needed so that the proposal response can be evaluated
       * within the context of the call's constraints.
       * @param proxy GEMSSProxy bound to a service provider
       * @param strCFP call for proposal XML string
       */
      public void setCFP( GEMSSProxy proxy, String strCFP ) throws Exception;

      /**
       * Add a proposal, tagged to a time stamp and participant. Normally the time stamp will be the current iteration (round)
       * of the auction.
       * @param proxy GEMSSProxy bound to a service provider
       * @param strWSLA string representation of a web service leval agreement
       * @param nTimeStamp time stamp
       */
      public void addProposal( GEMSSProxy proxy, String strWSLA, long nTimeStamp ) throws Exception;

      /**
       * Set the bidding finished flag for a party. A finished bidder will not be sent any more CFP's
       * @param proxy GEMSSProxy bound to a service provider
       */
      public void setFinished( GEMSSProxy proxy ) throws Exception;

      /**
       * Query the bidding finished flag for a party. A finished bidder will not be sent any more CFP's
       * @param proxy GEMSSProxy bound to a service provider
       */
      public boolean queryFinished( GEMSSProxy proxy ) throws Exception;

      /**
       * Sets the current bidding threshold
       * @param nThreshold new bidding threshold
       */
      public void setBiddingThreshold( double nThreshold ) throws Exception;

      /**
       * Sets the current deadline for proposals (if any)
       * @param nDeadline time offset from 2000 GTM
       */
      public void setProposalDeadline( long nDeadline ) throws Exception;

      /**
       * Sets the auction deadline, beyond which the auction must not go
       * @param nDeadline time offset from 2000 GTM
       */
      public void setAuctionDeadline( long nDeadline ) throws Exception;

      /**
       * get all parties
       * @return vector of GEMSSProxy objects
       */
      public Vector getParties() throws Exception;

      /**
       * get all proposals at a given time stamp
       * @param nTimeStamp time stamp
       * @return hash of (GEMSSProxy,WSLA string)
       */
      public Hashtable getProposals( long nTimeStamp ) throws Exception;

      /**
       * get WSLA template for a participant (null if not available)
       * Exceptions are passed on through
       * @param proxy GEMSSproxy object representing a participant
       * @return WSLA object
       */
      public String getWSLATemplate( GEMSSProxy proxy ) throws Exception;

      /**
       * Get the active (last) call for proposals for this proxy. The call is needed so
       * that the proposal response can be evaluated within the context of the call's constraints.
       * @param proxy GEMSSProxy bound to a service provider
       * @return call for proposal XML string, or null if there is none
       */
      public String getCFP( GEMSSProxy proxy ) throws Exception;

      /**
       * Returns the current bidding threshold
       * @return new bidding threshold
       */
      public double getBiddingThreshold() throws Exception;

      /**
       * Get the current deadline for proposals (if any) or -1 if not
       * @return time offset from 2000 GTM
       */
      public long getProposalDeadline() throws Exception;

      /**
       * Get the auction deadline or -1 if none
       * @return time offset from 2000 GTM
       */
      public long getAuctionDeadline() throws Exception;

      /**
       * Updates the timestamp, incrementing it by one, and returns the new value.
       * The timestamp is a simple number 1...N representing the bidding round, and will be checked against MaxRounds
       * @return new timestamp
       */
      public long updateTimestamp() throws Exception;

      /**
       * Returns the last timestamp. There will be timestamps from 1 .. current timestamp.
       * The timestamp is a simple number 1...N representing the bidding round.
       * @return current timestamp
       */
      public long getLastTimestamp() throws Exception;

}