/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package eu.gemss.components.negotiation;

import eu.gemss.components.Terminable;
import eu.gemss.components.negotiation.ProposalHistory;
import eu.gemss.components.negotiation.QoSRequirements;
import eu.gemss.components.proxies.GEMSSProxy;
import eu.gemss.GridException;

import java.util.Vector;


/**
 * This interface defines the functions supported by the QoSNegotiation component. This component is a client-side
 * component to drive a quality of service auction with several service providers. There must be a server side
 * module to respond to the invocations this module will generate, using the transport component.
 */
public interface QoSNegotiation {

	/**
     * Informs a set of service providers (represented by GEMSSProxy objects) that an auction is starting. The
	 * service providers will respond by getting ready to receive a call for proposal.
	 * @param strRequestDescFilename filename to the request desc file for the job under negotiation
	 * @param vectorProxies vector of GEMSSProxy objects with which to negotiate
     * @return Proposal history, which will initially contain only the particiant list since no bids have been made yet.
     */
	public ProposalHistory startAuction( String strRequestDescFilename, Vector vectorProxies ) throws GridException;

    /**
     * Get, by throwing a signal, a set of QoS requirements for this job. The requirements include QoS parameter
	 * weights and minimum acceptable values.
	 * @param strServiceName name of the service for which these requirements apply
	 * @param strJobDescription job description meaningful to the signal handler (will be passed via signal to the handler)
       * @param history Proposal histroy which includes details of the participants in the auction
     * @return a set of QoS requirements
     */
	public QoSRequirements getQoSRequirements( String strServiceName, String strJobDescription, ProposalHistory history ) throws GridException;

    /**
     * Sends a call for proposals message to each of the participants in the auction.
	 * @param requirements QoS requirements that will make up the detail of the cfp
	 * @param history Proposal histroy which includes details of the participants in the auction
       * @return Updated proposal history containing the new cfps
     */
	public ProposalHistory sendCFPs( QoSRequirements requirements, ProposalHistory history ) throws GridException;

    /**
     * Gets each of the proposals, if available, from the participants of the auction.
	 * @param history Proposal histroy which includes details of the participants in the auction
	 * @return Updated proposal history containing the new bids from service providers who responded in time
     */
	public ProposalHistory getProposals( ProposalHistory history ) throws GridException;

    /**
     * Evaluates a proposal history, and determins if the auction should finish or run another round of bidding.
	 * @param history Proposal histroy which includes details of the participants in the auction
	 * @return boolean return, true means end the auction, false means run another round of bidding
     */
	public boolean evaluateProposals( ProposalHistory history ) throws GridException;

    /**
     * Accepts the best proposal and returns a bound GEMSSProxy object ready for job handling. The proposal history
	 * is run from best bid to worst acceptable bid until a service provider successfully exchanges contracts. Only
	 * after signed digital contracts are exchanges is any party committed to thier agreeemnt. Once exchanged the other
	 * parties are rejected and the bound GEMSSProxy object returned.
	 * @param history Proposal histroy which includes details of the participants in the auction
	 * @return A bound GEMSSProxy object with which job handling can commence.
     */
	public GEMSSProxy acceptBestProposal( ProposalHistory history ) throws GridException;
}
