/*
 ************************************************************************
 *
 * Copyright (c) 2003-2004, C&C Research Laboratories, NEC Europe Ltd.
 *
 * Copyright in this software belongs to C&C Research Laboratories,
 * Rathausallee 10, 53757 Sankt Augustin, Germany.
 *
 * This software may not be used, sold, licensed, transferred, copied
 * or reproduced in whole or in part in any manner or form or in or
 * on any media by any person other than in accordance with the terms
 * of the Licence Agreement supplied with the software, or otherwise
 * without the prior written consent of the copyright owners.
 *
 * This software is distributed WITHOUT ANY WARRANTY, without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE, except where stated in the Licence Agreement supplied with
 * the software.
 *
 *  Created By :           G.A. Kohring
 *  Created for Project :  GEMSS (IST-2001-37153)
 *
 ************************************************************************
 *
 */

package eu.gemss.components.providers;

import java.io.IOException;
import java.util.Locale;
import eu.gemss.components.ServiceRequest;

/**
 * A superinterface for functionality common to all component providers.
 * <P>
 * A <em>component</em> is a set of interfaces and/or abstract classes.
 * A <em>component provider</em> is a lightweight factory providing a specific 
 * implementation of a <em>component</em>.  Since not
 * every component will be used in each application, it is important that
 * the provider class be free of any complex, time consuming initialization
 * procedures, otherwise the framework's initialization phase will be delayed.
 * <p>
 * The class implementing the provider interface must have a zero argument
 * constructor.
 * <p>
 * Component providers are identified to the system by placing a special
 * configuration file inside the Jar file containing the component. More
 * specifically, the file is placed in the: META-INF/services directory. 
 * The name of the file is the fully qualified name of the component, i.e.,
 * the component's interface. The file itself consists of the fully qualified
 * name of each provider class located in that Jar file. Each provider class
 * should be on a separate line and comments begin with the character
 * '#' (0x23). (All characters following the comment character are ignored.)
 * There can be multiple provider classes supporting multiple components 
 * within a single Jar file.
 * <p>
 * <strong>Example</strong><p>
 * Consider a provider for the <code>SecurityContext</code>, called 
 * <code>uk.ac.soton.itinnovation.SecurityContextProvider</code>. The Jar
 * file containing this component would have an entry: <p>
 * <code>META-INF/services/eu.gemss.component.SecurityContext</code> <p>
 * This file in turn, might contain the following lines: <p>
 * <listing>
 * # It Innovation's security context
 * uk.ac.soton.itinnovation.SecurityContextProvider
 * </listing>
 *
 */
public interface Provider {
    /**
     * Returns a non-<tt>null</tt> string identifying the name of the 
     * owner of this 
     * provider and the associated components it implements.
     *
     * @return a non-<code>null</code> <code>String</code> giving
     *      provider's name.
     */
    public String getProviderName();

    /**
     * Returns a string identifying the version of the component
     * supported by this provider.
     *
     * @return a <tt>String</tt> identifying
     *      the version of the component supported by this provider.
     */
    public String getVersion();

    /**
     * Ask whether the component will run locally or remotely.
     *
     * @return <tt>true</tt> if the component will run locally, or 
     *      <tt>false</tt> if it runs on a remote machine.
     */
    public boolean isLocal();

    /**
     * Returns a human readable description of the specified component.
     *
     * @return a string describing the implementation of this component,
     *      or <tt>null</tt> if this provider does not support this component.
     */
    public String getDescription( );

    /**
     * Returns a string identifying the component supported by
     * this provider.
     *
     * @return a <code>String</code> containing the fully qualified name of
     *      the component supported by this provider.
     */
    public String getComponentName();

    /**
     * Determine whether or not this provider supports the specified
     * component.
     *
     * @return <tt>true</tt> if this provider supports the specified component
     *      or <tt>false</tt> otherwise.
     */
    public boolean supports( String component );

    /**
     * Creates an instance of the component supported by this provider.
     *
     * @return an <tt>Object</tt> implementing the component
     *      interface. 
     * @throws IOException if an error occurs during loading,
     *      initialization, instantiation.
     */
    public Object createComponent( ) throws IOException;

    /**
     * Creates a component instance using the specified 
     * service request.
     *
     * @param sr a {@link eu.gemss.components.ServiceRequest} object
     *      containing detailed information regarding the service being
     *      being requested.
     * @return an <tt>Object</tt> implementing the specified component
     *      interface.
     * @throws IOException if an error occurs during loading,
     *      initialization, instantiation.
     */
    public Object createComponent( ServiceRequest sr ) throws IOException;
}
