/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: ISS, NEC and IT Innovation jointly designed this
//      interface class as part of the GEMSS project design discussions.
//      The code itself was written by IT Innovation alone.
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package eu.gemss.components.proxies;

import eu.gemss.GridException;

/**
 * This interface defines the set of common negotiation methods which all GEMSS services
 * are required to implement.
 */
public interface GEMSSNegotiationProxy {
      // Proxy flags
      public final static String mstaticNegPrepFlag = "GEMSSProxy-NegPreperationComplete";
      public final static String mstaticNegCompleteFlag = "GEMSSProxy-NegotiationComplete";
      public final static String mstaticJobPrepFlag = "GEMSSProxy-JobPerperationComplete";
      public final static String mstaticUploadRequestDescCompleteFlag = "GEMSSProxy-UploadRequestDescComplete";
      public final static String mstaticContractAgreedFlag = "GEMSSProxy-ContractAgreed";

      /**
       * Returns the job conversation ID spawned by the negotiation conversation after a successful
       * exchange of contracts and a moved to the job handling phase. The neg conv ID must be setup before
       * this method is called.
       * @return job conversation ID value returned by the service
       */
      public String getJobCID() throws GridException;

      /**
       * Returns the neg conversation ID spawned by the account conversation. The account conv ID is passed via the
       * constructor.
       * @return neg conversation ID value returned by the service
       */
      public String getNegCID() throws GridException;

      /**
       * get WSLA file for a specific account and service. This WSLA will reference the written agreement between
       * the client and service provider, and the account ref they have. It will also specify the QoS parameters
       * which the service provider will be expecting to be negotiated over.
       * @return string containing the WSLA object for this service
       */
      public String getWSLA() throws GridException;

      /**
       * Uploads the RequestDesc file. This is needed as a pre-requisit to starting the auction
       * since the service providers must run thier performance models to determine how long
       * the job will take to run.
       * @param strRequestDescFilename filename of the request desc file
       */
      public void uploadRequestDesc( String strRequestDescFilename ) throws GridException;

      /**
       * Inform function used to inform the service provider about the start of an auction
       * and the end of an auction.
       * @param strMessage FIPA message string containing the inform ACL message
       */
      public void auctionInform( String strMessage ) throws GridException;

      /**
       * Call for proposals function used to request from the the service provider a
       * proposal. Within the CFP is the deadline after which any proposals must be ready.
       * @param strMessage FIPA message string containing the CFP ACL message
       */
      public void auctionCFP( String strMessage ) throws GridException;

      /**
       * Propose function used to obtain a proposal from the service provider, in response to
       * a previous CFP.
       * @return FIPA message string containing the Propose ACL message
       */
      public String auctionPropose() throws GridException;

      /**
       * Accepts a proposal. The service provider will next expect to exchange signed contracts using
       * the request method. No commitment is made until contracts are exchanged.
       * @param strMessage FIPA message string containing the AcceptProposal ACL message
       */
      public void auctionAcceptProposal( String strMessage ) throws GridException;

      /**
       * Rejects a proposal. The service provider can now release any temporary reservations.
       * @param strMessage FIPA message string containing the RejectProposal ACL message
       */
      public void auctionRejectProposal( String strMessage ) throws GridException;

      /**
       * The client requests a signed contract, providing a signed contract itself. This allows for
       * the exchange of contracts.
       * @param strMessage FIPA message string containing the Request ACL message
       * @return FIPA message string containing the Inform ACL message
       */
      public String auctionRequest( String strMessage ) throws GridException;
}
