/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Darren Marvin
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package eu.gemss.components.security.token.types.pki;

import eu.gemss.components.security.token.TokenDescriptor;
import eu.gemss.components.security.token.types.key.asymmetric.AsymmetricKeyDescriptor;

/**
 * PKITokenType is a concrete implementation of a type of TokenID.
 * It concentrates on identifying types of PKI security token.
 * Distinguishing properties include cypher algorithm and key length.
 */
public class CertificateDescriptor implements TokenDescriptor {

    /**
     * PKI type for X.509 v3 as defined by RFC2549
     */
    public static int PKI_X509V3 = 0;
    /**
     *PKI type for PGP
     */
    public static int PKI_PGP = 1;

    private int mPKIType;
    private AsymmetricKeyDescriptor mKeyDesc;
    private static final String mDescription = "PKI Certificate Token";

    /**
     * Constructor creates a default Certificate that will use RSA
     */
    public CertificateDescriptor() {
        mPKIType = PKI_X509V3;
        mKeyDesc = new AsymmetricKeyDescriptor();
    }

    /**
     * Constructor allows the setting of a particular pki, cypher type
     * @param PKIType type of PKI
     * @param cypherAlg cypher algorithm
     */
    public CertificateDescriptor(int PKIType, int cypherAlg) {
        mPKIType = PKIType;
        mKeyDesc = new AsymmetricKeyDescriptor(AsymmetricKeyDescriptor.PUBLIC_KEY,cypherAlg);
    }

    /**
     * Check if this descriptor matches the passed one
     * @param tokenDesc token descriptor to check against
     * @return true if passed descriptor is the same both in terms of type and value, false otherwise
     */
    public boolean isEqual(TokenDescriptor tokenDesc) {

        if(tokenDesc instanceof CertificateDescriptor) {
            CertificateDescriptor cD = (CertificateDescriptor) tokenDesc;
            if(cD.getPKIType()==mPKIType && cD.getKeyDescriptor().isEqual(mKeyDesc))
                return true;

        }
        return false;
    }

    /**
     * Retrieve the PKI type for the token type
     * @return PKI type
     */
    public final int getPKIType() {
        return mPKIType;
    }

    /**
     * Retrieve the asymmetric key descriptor for this certificate. It describes the
     * properties of the public key for this certificate
     * @return asymmetric key descriptor
     */
    public final AsymmetricKeyDescriptor getKeyDescriptor() {
        return mKeyDesc;
    }

    public final String getDescription() {
        return mDescription;
    }

}


