/*
 ************************************************************************
 *
 * Copyright (c) 2003-2004, C&C Research Laboratories, NEC Europe Ltd.
 *
 * Copyright in this software belongs to C&C Research Laboratories,
 * Rathausallee 10, 53757 Sankt Augustin, Germany.
 *
 * This software may not be used, sold, licensed, transferred, copied
 * or reproduced in whole or in part in any manner or form or in or
 * on any media by any person other than in accordance with the terms
 * of the Licence Agreement supplied with the software, or otherwise
 * without the prior written consent of the copyright owners.
 *
 * This software is distributed WITHOUT ANY WARRANTY, without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE, except where stated in the Licence Agreement supplied with
 * the software.
 *
 *  Created By :           G.A. Kohring
 *  Created for Project :  GEMSS (IST-2001-37153)
 *
 ************************************************************************
 *
 */

package eu.gemss.components.util;

import java.util.zip.ZipFile;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.zip.ZipException;
import java.io.File;
import java.util.Enumeration;
import java.io.InputStream;
import java.util.zip.ZipEntry;

/**
 * An interface for reading archives in the ZIP format.
 */
public interface ZipArchive {

    /**
     * Lists the names of all the entries in the archive.
     *
     * @return an array of <code>String</code>s containing the names of all
     *      the entries in the archive. An empty archive has an array of
     *      zero length.
     */
    public String[] getEntryNames();

    /**
     * Extracts the specified entry from the archive and places it into 
     * the specified directory.
     *
     * @param entryName a <code>String</code> containing the name of the entry
     *      which should be extracted.
     * @param destination a <code>File</code> object naming a directory into
     *      which the archive should be unpacked.
     * @throws FileNotFoundException if 
     *      the specified destination is not a directory.
     * @throws IOException if an I/O error occurs.
     * @throws ZipException if the specified entry does not exist in this
     *      archive or if any other Zip file error occurs.
     */
    public void extract( String entryName, File destination )
            throws FileNotFoundException, IOException, ZipException;

    /**
     * Extracts the contents of the archive into the specified directory.
     *
     * @param destination a <code>File</code> object naming a directory into
     *      which the archive should be unpacked.
     * @throws FileNotFoundException if 
     *      the specified destination is not a directory.
     * @throws IOException if an I/O error occurs.
     * @throws ZipException if any Zip file error occurs.
     */
    public void extractAll(  File destination ) 
            throws FileNotFoundException, IOException, ZipException;

    /**
     * Retrieve the number of entries in this archive.
     *
     * @return an <code>int</code> representing the number of entries in this
     *      archive.
     */
    public int size();

    /**
     * Close this archive. No further operations are possible once the archive
     * is closed.
     *
     * @throws IOException if an error occurs while closing the archive.
     */
    public void close() throws IOException ;

    /**
     * Retrieves the name of this archive.
     *
     * @return a <code>String</code> containing the name of the archive file.
     */
    public String getName();

    /**
     * List all the entries in this archive.
     *
     * @return an <code>Enumeration</code> of all entries in this archive.
     */
    public Enumeration entries();

    /**
     * Retrieves and input stream for the specified entry.
     *
     * @param entry a <code>ZipEntry</code> object the entry whose input
     *      stream is to be returned.
     * @return an <code>InputStream</code> for the specified entry.
     * @throws IOException if an I/O error occurs while trying to open an
     *      input stream on the specified entry.
     */
    public InputStream getInputStream( ZipEntry entry ) throws IOException ;

    /**
     * Retrieves a <code>ZipEntry</code> object for the specified entry.
     *
     * @param entryName a <code>String</code> identifying the entry.
     * @return a <code>ZipEntry</code> object for the specified entry.
     */
    public ZipEntry getEntry( String entryName );
}
