/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.negotiation.auction;

import java.lang.*;

import uk.ac.soton.itinnovation.gemss.negotiation.auction.AuctionMessageBody;
import uk.ac.soton.itinnovation.gemss.negotiation.auction.AuctionMessageHeader;
import uk.ac.soton.itinnovation.gemss.negotiation.auction.AuctionMessage;

/*
 * AuctionMessageHandler interface. Classes that implement this interface provide the basic functionality to format and
 * handle auction type messages. This interface is both message format and transport format independant.
 * These message types are based on the FIPA English auction protocol (FIPA English Auction Interaction Protocol Specification XC00031F)
 * This interface hides the actual FIPA syntax, so the English auction protocol can be used by non-FIPA messaging syntaxes
 * if the implementing class wants to use an alternative syntax
*/
public interface AuctionMessageHandler {

      // auction message type constants
      public static final int mstaticUnknownType = 0;
      public static final int mstaticInformType = 1;
      public static final int mstaticCFPType = 2;
      public static final int mstaticProposeType = 3;
      public static final int mstaticAcceptProposalType = 4;
      public static final int mstaticRejectProposalType = 5;
      public static final int mstaticRequestType = 6;

	/**
	 * Auction infrom. Used to inform about the start of an auction, respond to a request for a signed contract and
       * inform about the end of an auction. The body of the inform message contains info about the type of the inform
       * message, and will use the syntax declared in language.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatInformMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception;

	/**
	 * Auction call for participation message. This message is used to request a proposal from each particiant, and
	 * forms the basis for a bid. The content is typically a conplex description of the initiators bidding criteria,
	 * such as minimum values and  importance weightings on bid properties.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatCFPMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception;

	/**
	 * Auction propose message. This message is used to respond to a CFP message request with an actual bid.
	 * The content is typically a conplex description of the participants bid properties.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatProposeMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception;

	/**
	 * Auction accept proposal message. This message is used to respond to a propose message and accept it. This might
	 * trigger a request message to ask for a signed contract, or some concluding auction document from the participant.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatAcceptProposalMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception;

	/**
	 * Auction reject proposal message. This message is used to respond to a propose message and reject it. Rejection is normall
	 * followed by an inform to close the auction.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatRejectProposalMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception;

	/**
	 * Auction request contract message. This message is normally send after a proposal is agreed.
	 * The content of this message is normally a signed contract for the participant's records.
	 * The initiator will expect a informContract message to be sent back next.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatRequestContractMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception;

	/**
	 * Extract the message header from a message
	 * @param message message to parse
	 * @return message header object parsed from the message
	 */
	public AuctionMessageHeader parseMessageHeader( AuctionMessage message ) throws Exception;

	/**
	 * Extract the message body from a message
	 * @param message message to parse
	 * @return Message body object parsed from the message
	 */
	public AuctionMessageBody parseMessageBody( AuctionMessage message ) throws Exception;

      /**
       * Extract the message type from a message
       * @param message message to parse
       * @return Message type of this message
       */
      public int parseMessageType( AuctionMessage message ) throws Exception;

} // end of AuctionMessageHandlerImpl
