/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.negotiation.auction;

import java.lang.*;
import java.util.logging.Level;
import java.util.logging.Logger;

import uk.ac.soton.itinnovation.gemss.negotiation.auction.AuctionMessageBody;
import uk.ac.soton.itinnovation.gemss.negotiation.auction.AuctionMessageHeader;
import uk.ac.soton.itinnovation.gemss.negotiation.auction.AuctionMessage;

import fipaos.ont.fipa.ACL;
import fipaos.parser.acl.string.Parser;
import fipaos.ont.fipa.FIPACONSTANTS;
import fipaos.ont.fipa.fipaman.AgentID;

/*
 * AuctionMessageHandlerFIPAImpl class implementation. Ths class provides the basic functionality to format and
 * handle auction type messages. It is based on the FIPA English auction protocol, and provides support
 * for all the message types required for an English auction.
 * Details of the FIPA messaging formats used are abstracted away, and users need only transmit the FIPA
 * messages generated as simple strings. As such this class is transport protocol neutral.
*/
public class AuctionMessageHandlerFIPAImpl implements AuctionMessageHandler {

	// logger variable
	static private Logger mlogger = Logger.getLogger("uk.ac.soton.itinnovation.gemss.negotiation.auction.auctionmessagehandlerfipaimpl");

	/**
	 * formatInformMessage
       * FIPA infrom. Used to inform about the start of an auction, respond to a request for a signed contract and
       * inform about the end of an auction. The body of the inform message contains info about the type of the inform
       * message, and will use the syntax declared in language.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatInformMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception
	{
		ACL aclMessage;
		AuctionMessage message;
		String strSerializedForm;

		try {
			// basic params checks
			if( body == null || header == null ) throw new Exception("null parameters");

			// make a new FIPA object initialized from the message header information
			aclMessage = setACLMessageHeaderInfo( header );

			// setup content from the body and message type
			aclMessage.setContentObject( body.toString() );
			aclMessage.setPerformative( FIPACONSTANTS.INFORM );	// FIPA inform

			// serialize the FIPA object
			strSerializedForm = serializeFIPAMessage( aclMessage );

			// make an auction message
			message = new AuctionMessage( strSerializedForm );

			// all done
			return message;
		}
		catch ( Exception ex ) {
			mlogger.log( Level.SEVERE,"Exception during formatInformMessage",ex);
			throw new Exception("formatInformMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
		}
	}

	/**
	 * formatCFPMessage
	 * Auction call for participation message. This message is used to request a proposal from each particiant, and
	 * forms the basis for a bid. The content is typically a conplex description of the initiators bidding criteria,
	 * such as minimum values and  importance weightings on bid properties.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatCFPMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception
	{
		ACL aclMessage;
		AuctionMessage message;
		String strSerializedForm;

		try {
			// basic params checks
			if( body == null || header == null ) throw new Exception("null parameters");

			// make a new FIPA object initialized from the message header information
			aclMessage = setACLMessageHeaderInfo( header );

			// setup content from the body and message type
			aclMessage.setContentObject( body.toString() );
			aclMessage.setPerformative( FIPACONSTANTS.CFP );	// FIPA cfp

			// serialize the FIPA object
			strSerializedForm = serializeFIPAMessage( aclMessage );

			// make an auction message
			message = new AuctionMessage( strSerializedForm );

			// all done
			return message;
            }
            catch ( Exception ex ) {
                  mlogger.log( Level.SEVERE,"Exception during formatCFPMessage",ex);
                  throw new Exception("formatCFPMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
            }
	}

	/**
	 * formatProposeMessage
	 * Auction propose message. This message is used to respond to a CFP message request with an actual bid.
	 * The content is typically a conplex description of the participants bid properties.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatProposeMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception
	{
		ACL aclMessage;
		AuctionMessage message;
		String strSerializedForm;

		try {
			// basic params checks
			if( body == null || header == null ) throw new Exception("null parameters");

			// make a new FIPA object initialized from the message header information
			aclMessage = setACLMessageHeaderInfo( header );

			// setup content from the body and message type
			aclMessage.setContentObject( body.toString() );
			aclMessage.setPerformative( FIPACONSTANTS.PROPOSE );	// FIPA propose

			// serialize the FIPA object
			strSerializedForm = serializeFIPAMessage( aclMessage );

			// make an auction message
			message = new AuctionMessage( strSerializedForm );

			// all done
			return message;
            }
            catch ( Exception ex ) {
                  mlogger.log( Level.SEVERE,"Exception during formatProposeMessage",ex);
                  throw new Exception("formatProposeMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
            }
	}

	/**
	 * formatAcceptProposalMessage
	 * Auction accept proposal message. This message is used to respond to a propose message and accept it. This might
	 * trigger a request message to ask for a signed contract, or some concluding auction document from the participant.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatAcceptProposalMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception
	{
		ACL aclMessage;
		AuctionMessage message;
		String strSerializedForm;

		try {
			// basic params checks
			if( body == null || header == null ) throw new Exception("null parameters");

			// make a new FIPA object initialized from the message header information
			aclMessage = setACLMessageHeaderInfo( header );

			// setup content from the body and message type
			aclMessage.setContentObject( body.toString() );
			aclMessage.setPerformative( FIPACONSTANTS.ACCEPT_PROPOSAL );	// FIPA accept proposal

			// serialize the FIPA object
			strSerializedForm = serializeFIPAMessage( aclMessage );

			// make an auction message
			message = new AuctionMessage( strSerializedForm );

			// all done
			return message;
            }
            catch ( Exception ex ) {
                  mlogger.log( Level.SEVERE,"Exception during formatAcceptProposalMessage",ex);
                  throw new Exception("formatAcceptProposalMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
            }
	}

	/**
	 * formatRejectProposalMessage
	 * Auction reject proposal message. This message is used to respond to a propose message and reject it. Rejection is normall
	 * followed by an inform to close the auction.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatRejectProposalMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception
	{
		ACL aclMessage;
		AuctionMessage message;
		String strSerializedForm;

		try {
			// basic params checks
			if( body == null || header == null ) throw new Exception("null parameters");

			// make a new FIPA object initialized from the message header information
			aclMessage = setACLMessageHeaderInfo( header );

			// setup content from the body and message type
			aclMessage.setContentObject( body.toString() );
			aclMessage.setPerformative( FIPACONSTANTS.REJECT_PROPOSAL );	// FIPA reject proposal

			// serialize the FIPA object
			strSerializedForm = serializeFIPAMessage( aclMessage );

			// make an auction message
			message = new AuctionMessage( strSerializedForm );

			// all done
			return message;
            }
            catch ( Exception ex ) {
                  mlogger.log( Level.SEVERE,"Exception during formatRejectProposalMessage",ex);
                  throw new Exception("formatRejectProposalMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
            }
	}

	/**
	 * formatRequestContractMessage
	 * Auction request contract message. This message is normally send after a proposal is agreed.
	 * The content of this message is normally a signed contract for the participant's records.
	 * The initiator will expect a informContract message to be sent back next.
	 * @param header AuctionMessageHeader object containing header information
	 * @param body AuctionMessageBody object containing the message content
	 * @return auction message formatted according to the implementing class's syntax
	 */
	public AuctionMessage formatRequestContractMessage( AuctionMessageHeader header, AuctionMessageBody body ) throws Exception
	{
		ACL aclMessage;
		AuctionMessage message;
		String strSerializedForm;

		try {
			// basic params checks
			if( body == null || header == null ) throw new Exception("null parameters");

			// make a new FIPA object initialized from the message header information
			aclMessage = setACLMessageHeaderInfo( header );

			// setup content from the body and message type
			aclMessage.setContentObject( body.toString() );
			aclMessage.setPerformative( FIPACONSTANTS.REQUEST );	// FIPA request

			// serialize the FIPA object
			strSerializedForm = serializeFIPAMessage( aclMessage );

			// make an auction message
			message = new AuctionMessage( strSerializedForm );

			// all done
			return message;
            }
            catch ( Exception ex ) {
                  mlogger.log( Level.SEVERE,"Exception during formatRequestContractMessage",ex);
                  throw new Exception("formatRequestContractMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
            }
	}

	/**
	 * parseMessageHeader
	 * Extract the message header from a FIPA message
	 * @param message FIPA message to parse
	 * @return message header object parsed from the FIPA message
	 */
	public AuctionMessageHeader parseMessageHeader( AuctionMessage message ) throws Exception
	{
		ACL aclMessage;
		AgentID agentid;
		String strSenderID, strReceiverID;
		AuctionMessageHeader header;

		try {
			// basic params checks
			if( message == null ) throw new Exception("null parameters");

			// parse the message
			aclMessage = deserializeFIPAMessage( message.toString() );

			// validate the message protocol
			if( !aclMessage.getProtocol().equals( FIPACONSTANTS.FIPA_AUCTION_ENGLISH ) ) throw new Exception("FIPA protocol not of an English auction");

			// log message type for interest (not used at present)
			mlogger.log( Level.INFO,"FIPA message type [body parse] = "+aclMessage.getPerformative() );

			// get the agent ID's
			agentid = aclMessage.getSenderAID();
			if( agentid == null ) throw new Exception("null sender ID");
			strSenderID = agentid.getName();

			agentid = aclMessage.getReceiverAID();
			if( agentid == null ) throw new Exception("null receiver ID");
			strReceiverID = agentid.getName();

			// get the message header
                  header = new AuctionMessageHeader(  aclMessage.getConversationID(),
                                                      aclMessage.getLanguage(),
                                                      aclMessage.getOntology(),
                                                      strSenderID,
                                                      strReceiverID );

			// all done
			return header;
		}
		catch ( Exception ex ) {
			mlogger.log( Level.SEVERE,"Exception during parseMessageHeader",ex);
			throw new Exception("parseMessageHeader failed: "+ex.getMessage()+" - see java Logger log for full details");
		}
	}

	/**
	 * parseMessageBody
	 * Extract the message body from a FIPA message
	 * @param message FIPA message to parse
	 * @return Message body object parsed from the FIPA message
	 */
	public AuctionMessageBody parseMessageBody( AuctionMessage message ) throws Exception
	{
		ACL aclMessage;
		AuctionMessageBody body;

		try {
			// basic params checks
			if( message == null ) throw new Exception("null parameters");

			// parse the message
			aclMessage = deserializeFIPAMessage( message.toString() );

			// get the message body
			body = new AuctionMessageBody( (String) aclMessage.getContentObject() );

			// all done
			return body;
		}
		catch ( Exception ex ) {
			mlogger.log( Level.SEVERE,"Exception during parseMessageBody",ex);
			throw new Exception("parseMessageBody failed: "+ex.getMessage()+" - see java Logger log for full details");
		}
	}

      /**
       * Extract the message type from a message
       * @param message message to parse
       * @return Message type of this message
       */
      public int parseMessageType( AuctionMessage message ) throws Exception
      {
            ACL aclMessage;
            String strType;
            int nType;

            try {
                  // basic params checks
                  if( message == null ) throw new Exception("null parameters");

                  // parse the message
                  aclMessage = deserializeFIPAMessage( message.toString() );

                  // get the message type
                  strType = (String) aclMessage.getPerformative();

                  // get it's flag value (default is unknown)
                  nType = AuctionMessageHandler.mstaticUnknownType;
                  if( strType.equals( FIPACONSTANTS.INFORM ) )
                        nType = AuctionMessageHandler.mstaticInformType;
                  if( strType.equals( FIPACONSTANTS.CFP ) )
                        nType = AuctionMessageHandler.mstaticCFPType;
                  if( strType.equals( FIPACONSTANTS.PROPOSE ) )
                        nType = AuctionMessageHandler.mstaticProposeType;
                  if( strType.equals( FIPACONSTANTS.ACCEPT_PROPOSAL ) )
                        nType = AuctionMessageHandler.mstaticAcceptProposalType;
                  if( strType.equals( FIPACONSTANTS.REJECT_PROPOSAL ) )
                        nType = AuctionMessageHandler.mstaticRejectProposalType;
                  if( strType.equals( FIPACONSTANTS.REQUEST ) )
                        nType = AuctionMessageHandler.mstaticRequestType;

                  // all done
                  return nType;
            }
            catch ( Exception ex ) {
                  mlogger.log( Level.SEVERE,"Exception during parseMessageType",ex);
                  throw new Exception("parseMessageType failed: "+ex.getMessage()+" - see java Logger log for full details");
            }
      }

	/**
	 * setACLMessageHeaderInfo
	 * Uses an AuctionMessageHeader object to initialize an ACLMessage. The message content and performative fields
	 * will still need to be set.
	 * @param header auction header object which will be used to initalize the ACL message
	 * @return FIPA ACL object
	 */
	private ACL setACLMessageHeaderInfo( AuctionMessageHeader header ) throws Exception
	{
		ACL aclMessage;
		AgentID agentid;

		try {
			// make a new FIPA object
			aclMessage = new ACL();

			// FIPA engligh auction constants
			aclMessage.setProtocol( FIPACONSTANTS.FIPA_AUCTION_ENGLISH );		// FIPA format english auction

			// use the header parameters to initialize the rest of the FIPA object
			aclMessage.setConversationID( header.getConversationID() );
			aclMessage.setLanguage( header.getLanguage() );
			aclMessage.setOntology( header.getOntology() );

			// make agent ID's out of the header provided ID's
			agentid = new AgentID();
			agentid.setName( header.getSenderID() );
			aclMessage.setSenderAID( agentid );

			agentid = new AgentID();
			agentid.setName( header.getReceiverID() );
			aclMessage.setReceiverAID( agentid );

			// return partially initialized ACL message
			return aclMessage;
		}
		catch ( Exception ex ) {
			mlogger.log( Level.SEVERE,"Exception during setACLMessageHeaderInfo",ex);
			throw new Exception("setACLMessageHeaderInfo failed: "+ex.getMessage()+" - see java Logger log for full details");
		}
	}

	/**
	 * serializeFIPAMessage
	 * Serializes a FIPA message object and returns a String containing the serialized text
	 * @param message FIPA ACL message which will be serialized
	 * @return string containing serialized message
	 */
	private String serializeFIPAMessage( ACL aclMessage ) throws Exception
	{
		Parser parserString;
		byte [] arrayBytes;
            String strText;

		try {
			// serialize it to a string
			parserString = new Parser();
			arrayBytes = parserString.deparse( aclMessage );

			// get the string
			strText = new String( arrayBytes );

                  // encode any special characters (none for now - just plain text with newlines, <> and "" etc)

                  // return string
                  return strText;
		}
		catch ( Exception ex ) {
			mlogger.log( Level.SEVERE,"Exception during serializeFIPAMessage",ex);
			throw new Exception("serializeFIPAMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
		}
	}

	/**
	 * deserializeFIPAMessage
	 * De-serializes a string and returns a FIPA message object
	 * @param message String to be parsed
	 * @return FIPA message object
	 */
	private ACL deserializeFIPAMessage( String strMessage ) throws Exception
	{
		Parser parserString;
		byte [] arrayBytes;
            String strText;

		try {
                  // decode any special characters (none for now - just plain text with newlines, <> and "" etc)
                  strText = strMessage;

			// de-serialize it to an ACL object
			arrayBytes = strText.getBytes();

                  // make a parser
                  parserString = new Parser();

                  // parse bytes
			return parserString.parse( arrayBytes );
		}
		catch ( Exception ex ) {
			mlogger.log( Level.SEVERE,"Exception during deserializeFIPAMessage",ex);
			throw new Exception("deserializeFIPAMessage failed: "+ex.getMessage()+" - see java Logger log for full details");
		}
	}

} // end of AuctionMessageHandlerFIPAImpl
