/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.proxies;

import java.util.logging.Logger;
import java.util.logging.Level;
import java.io.*;

import uk.ac.soton.itinnovation.gemss.proxies.*;
import eu.gemss.components.proxies.*;
import eu.gemss.components.providers.*;

/**
 * IT Innovation provider for the gemss proxy component. Each call to create a GEMSS
 * proxy will return a new instance.
 */
public class GEMSSProxyProvider implements Provider {

    private static Logger mLogger = Logger.getLogger("uk.ac.soton.itinnovation.gemss.proxies.gemssproxyprovider");
    private static final String GEMSS_PROXY_IDENTIFIER = "eu.gemss.components.proxies.GEMSSProxy";
    private static final String COMPONENT_DESCRIPTION = "GEMSS Proxy Component";
    private static final String GEMSS_PROXY_OWNER = "IT Innovation Centre";
    private static final boolean IS_LOCAL = true;
    private static final String VERSION = "0.1";

    /**
     * Default constructor takes no arguments so that it can be loaded easily
     * within the plug and play framework.
     */
    public GEMSSProxyProvider() {
    }

    /**
     * Creates an instance of the GEMSSProxy (custom method not part of Provider). This will
     * create a new instance of the GEMSSProxy component regardless of how many already
     * exist.
     * @param strConfigFile filename of the config file from which the operation / wsdl details can be read
     * @return instance of GEMSSProxy, initialized with the parameters passed to this method
     */
    public GEMSSProxy createGEMSSProxy( String strConfigFile ) throws Exception
    {
	    GEMSSProxyImp proxy;

	    try{
                // check param
                if( strConfigFile == null ) throw new Exception("null config file parameter");

		    //create a new instance
		    proxy = new GEMSSProxyImp();

		    // call init
		    proxy.init( strConfigFile );

		    // return the proxy
		    return proxy;
	    }
	    catch( Exception ex ) {
		    mLogger.log(Level.SEVERE,"GEMSSProxyProvider failed to create GEMSSProxyImp",ex);
		    throw new Exception("GEMSSProxyProvider failed to create GEMSSProxyImp");
	    }
    }

    /**
     * Creates an instance of the GEMSSProxy and initializes it.
     * @param serviceRequest service request where init info can be passed
     * @return instance of GEMSSProxy, initialized with the infromation contained in the service request.
     * If the service request is not of type GEMSSProxyRequest then no initialization will be done - this
     * should be considered an error since a GEMSSProxy with no initialization is useless.
     */
    public Object createComponent( eu.gemss.components.ServiceRequest serviceRequest ) throws IOException
    {
	    GEMSSProxyImp proxy;
	    GEMSSProxyRequest request;

	    try{
                // check for a null service request
                if( serviceRequest == null ) throw new Exception("null serviceRequest parameter");

		    // make proxy
		    proxy = new GEMSSProxyImp();

		    // set init parameters IF the right service request is used
		    if( serviceRequest instanceof GEMSSProxyRequest ) {
			    request = (GEMSSProxyRequest) serviceRequest;
                      if( request.getConfigFile() != null ) {
                            proxy.init( request.getConfigFile() );
                      }
                      else {
                            proxy.init( request.getWSDLURI(),
                                        request.getServiceName(),
                                        request.getEndPointPortName(),
                                        request.getServicePortType(),
                                        request.getBinding(),
                                        request.getOpUploadData(),
                                        request.getOpStart(),
                                        request.getOpKill(),
                                        request.getOpGetStatus(),
                                        request.getOpDownload(),
                                        request.getOpAcknowledgeResults(),
                                        request.getOpGetWSLA(),
                                        request.getOpUploadRequestDesc(),
                                        request.getOpAuctionInform(),
                                        request.getOpAuctionCFP(),
                                        request.getOpAuctionPropose(),
                                        request.getOpAuctionAcceptProposal(),
                                        request.getOpAuctionRejectProposal(),
                                        request.getOpAuctionRequest(),
                                        request.getOpGetJobCID(),
                                        request.getOpGetNegCID(),
                                        request.getArgConversationID(),
                                        request.getArgFilenameParam(),
                                        request.getArgInputDataFileTag(),
                                        request.getArgOutputDataFileTag(),
                                        request.getArgAuctionMessage(),
                                        request.getUsePolicy() );
                      }
		    }
		    else throw new Exception( "not passed a GEMSSProxyRequest object, cannot initialize proxy" );

		    // create a new instance and do NOT call init
		    return proxy;
	    }
	    catch(Exception ex) {
		    mLogger.log(Level.SEVERE,"GEMSSProxyProvider failed to create GEMSSProxyImp",ex);
		    throw new IOException("GEMSSProxyProvider failed to create GEMSSProxyImp");
	    }
    }

    /**
     * Creates an instance of the GEMSSProxy.
     * @return instance of GEMSSProxy, not initialized in any way - this
     * should be considered an error since a GEMSSProxy with no initialization is useless.
     */
    public Object createComponent() throws IOException {
	  try{
		  // create a new instance and do NOT call init
		  mLogger.log(Level.INFO,"No arguments passed to createComponent, GEMSSProxy not initialized");
		  return new GEMSSProxyImp();
	  }
	  catch(Exception ex) {
		mLogger.log(Level.SEVERE,"GEMSSProxyProvider failed to create GEMSSProxyImp",ex);
		throw new IOException("GEMSSProxyProvider failed to create GEMSSProxyImp");
	  }
    }

    public boolean supports(String identifier) {
        if(identifier.equals(GEMSS_PROXY_IDENTIFIER))
            return true;
        else
            return false;
    }

    public String getComponentName() {
        return GEMSS_PROXY_IDENTIFIER;

    }

    public String getProviderName() {
	  return GEMSS_PROXY_OWNER;

    }

    public String getDescription() {
        return COMPONENT_DESCRIPTION;
    }

    public boolean isLocal() {
        return IS_LOCAL;
    }

    public String getVersion() {
        return VERSION;
    }
}