/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Darren Marvin
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.security.context;

import java.util.logging.Logger;
import java.util.logging.Level;

import eu.gemss.signals.*;
import eu.gemss.components.security.token.*;
import eu.gemss.components.security.token.types.*;
import eu.gemss.components.security.*;


import uk.ac.soton.itinnovation.gemss.security.context.configuration.*;
import uk.ac.soton.itinnovation.gemss.security.context.token.SecurityTokenManager;
import uk.ac.soton.itinnovation.gemss.utils.configuration.*;

/**
 * SecurityContext provides access to the security context that applies to this application.
 * It can retrieve security tokens that the application can use in interactions with other parties.
 * It can verify trust for security tokens generated elsewhere. It does all this without burdening
 * the client about the details of the security context implementation and local security infrastructure.
 */
public class SecurityContext {

    private Logger mLogger = Logger.getLogger("uk.ac.soton.itinnovation.gemss.security.context.SecurityContext");
    private SecurityContextConfiguration mConfiguration = null;
    private SecurityTokenManager mSecurityTokenManager = null;

    /**
     * Default constructor will use default values for configuration options
     * @throws SecurityContextException
     */
    public SecurityContext() throws SecurityContextException {
        try{
            init();
            //create a SecurityTokenManager with default signal listener
            mSecurityTokenManager = new SecurityTokenManager(mConfiguration,new StaticConfigurationSignalHandler());
        }
        catch(eu.gemss.components.security.token.SecurityTokenException ex) {
            throw new SecurityContextException(ex.getMessage());
        }
        catch(Exception ex) {
            mLogger.log(Level.SEVERE,"Unexplained failure, please send a bug report including all log files",ex);
            throw new SecurityContextException("Unexplained failure, please send a bug report including all log files");
        }
    }


    /**
     * Constructor enables a security context to use the supplied signal processor for processing
     * signals / queries from underlying security infrastructure
     * @param handler signal handler for processing signals
     * @throws SecurityContextException
     */
    public SecurityContext(SignalHandler handler) throws SecurityContextException {
        try{
            init();
            //create a SecurityTokenManager with the supplied listener
            mSecurityTokenManager = new SecurityTokenManager(mConfiguration,handler);
        }
        catch(eu.gemss.components.security.token.SecurityTokenException ex) {
            throw new SecurityContextException(ex.getMessage());
        }
        catch(Exception ex) {
            mLogger.log(Level.SEVERE,"Unexplained failure, please send a bug report including all log files",ex);
            throw new SecurityContextException("Unexplained failure, please send a bug report including all log files");
        }
    }

    /**
     * Constructor enables a security context to use the supplied configuration.
     * @param config configuration to adopt
     * @throws SecurityContextException
     */
    public SecurityContext(SecurityContextConfiguration config) throws SecurityContextException {
        try{
            init(config);
            //create a SecurityTokenManager with default signal listener
            mSecurityTokenManager = new SecurityTokenManager(mConfiguration,new StaticConfigurationSignalHandler());
        }
        catch(eu.gemss.components.security.token.SecurityTokenException ex) {
            throw new SecurityContextException(ex.getMessage());
        }
        catch(Exception ex) {
            mLogger.log(Level.SEVERE,"Unexplained failure, please send a bug report including all log files",ex);
            throw new SecurityContextException("Unexplained failure, please send a bug report including all log files");
        }
    }

    /**
     * Constructor enables a security context to use the supplied signal processor for processing
     * signals / queries from underlying security infrastructure and to adopt the supplied configuration
     * @param config configuration to adopt
     * @param handler signal handler for processing signals
     * @throws SecurityContextException
     */
    public SecurityContext(SecurityContextConfiguration config,SignalHandler handler) throws SecurityContextException {
        try{
            init(config);
            //create a SecurityTokenManager with the supplied listener
            mSecurityTokenManager = new SecurityTokenManager(mConfiguration,handler);
        }
        catch(eu.gemss.components.security.token.SecurityTokenException ex) {
            throw new SecurityContextException(ex.getMessage());
        }
        catch(Exception ex) {
            mLogger.log(Level.SEVERE,"Unexplained failure, please send a bug report including all log files",ex);
            throw new SecurityContextException("Unexplained failure, please send a bug report including all log files");
        }
    }

    /**
     * Generate a security token for the passed token descriptor, it is not automatically stored in the context
     * Calling this method several times could produce different security tokens.
     * @param tokenDesc token descriptor
     * @return security token
     * @throws SecurityContextException
     * @throws UnsupportedTokenException
     */
    public SecurityToken generateSecurityToken(TokenDescriptor tokenDesc) throws SecurityContextException, UnsupportedTokenException {
        try{
            return mSecurityTokenManager.generateSecurityToken(tokenDesc);
        }
        catch(Exception ex) {
            if(ex instanceof SecurityContextException) {
                throw (SecurityContextException) ex;
            }
            mLogger.log(Level.SEVERE,"Unexplained failure in security context: '" + ex.getMessage() + "'",ex);
            throw new SecurityContextException("Unexplained failure in security context: '" + ex.getMessage() + "'");
        }
    }

    /**
     * Check to see if the supplied security token is trusted within the prevailing security context
     * @param secToken security token to check
     * @return true if trusted, false otherwise
     * @throws SecurityContextException
     */
    public boolean isTrusted(SecurityToken secToken) throws SecurityContextException {
        try{
            return mSecurityTokenManager.isTrusted(secToken);
        }
        catch(Exception ex) {
                if(ex instanceof SecurityContextException) {
                    throw (SecurityContextException) ex;
                }
                mLogger.log(Level.SEVERE,"Unexplained failure in security context: '" + ex.getMessage() + "'",ex);
                throw new SecurityContextException("Unexplained failure in security context: '" + ex.getMessage() + "'");
            }
    }

    /**
     * Add a security token instance to the prevailing security context
     * @param secToken security token to add
     * @throws SecurityContextException
     */
    public void addSecurityTokenToContext(SecurityToken secToken) throws SecurityContextException {
        try{
            mSecurityTokenManager.addSecurityTokenToContext(secToken);
        }
        catch(Exception ex) {
            if(ex instanceof SecurityContextException) {
                throw (SecurityContextException) ex;
            }
            mLogger.log(Level.SEVERE,"Unexplained failure in security context: '" + ex.getMessage() + "'",ex);
            throw new SecurityContextException("Unexplained failure in security context: '" + ex.getMessage() + "'");
        }
    }

    /**
     * Remove from the prevailing security context the security token identified by the supplied token descriptor
     * @param secToken token to remove
     * @throws SecurityContextException
     */
    public void removeSecurityTokenFromContext(SecurityToken secToken) throws SecurityContextException {
        try{
            mSecurityTokenManager.removeSecurityTokenFromContext(secToken);
        }
        catch(Exception ex) {
            if(ex instanceof SecurityContextException) {
                throw (SecurityContextException) ex;
            }
            mLogger.log(Level.SEVERE,"Unexplained failure in security context: '" + ex.getMessage() + "'",ex);
            throw new SecurityContextException("Unexplained failure in security context: '" + ex.getMessage() + "'");
        }
    }

    /**
     * Retrieve from the prevailing security context a previously stored security token matching the passed token descriptor
     * @param tokenDesc token identifier
     * @return security token required
     * @throws SecurityContextException
     */
    public SecurityToken getSecurityTokenFromContext(TokenDescriptor tokenDesc) throws SecurityContextException {
        try{
            return mSecurityTokenManager.getSecurityTokenFromContext(tokenDesc);
        }
        catch(Exception ex) {
            if(ex instanceof SecurityContextException) {
                throw (SecurityContextException) ex;
            }
            mLogger.log(Level.SEVERE,"Unexplained failure in security context: '" + ex.getMessage() + "'",ex);
            throw new SecurityContextException("Unexplained failure in security context: '" + ex.getMessage() + "'");
        }
    }

    private void init() throws SecurityContextException {
        try{
        //grab the configuration
        mConfiguration = new SecurityContextConfiguration();
        }
        catch(ConfigurationException ex) {
            mLogger.log(Level.SEVERE,"Could not configure the security context, more information in log files",ex);
            throw new SecurityContextException("Could not configure the security context, more information in log files");
        }
    }

    private void init(SecurityContextConfiguration config) throws SecurityContextException {
        //grab the configuration
        mConfiguration = config;
    }

    private class StaticConfigurationSignalHandler implements SignalHandler {

        /**
         * Accepts signal and forwards them to the true signal
         * handler for the software. This should be blocking but does depend
         * on how the particular signal framework for the software is implemented.
         */
        public void process(Signal signal) {
            //this implementation does nothing.
            //it throws an exception to indicate that no response is coming

        }

    }
}