/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Darren Marvin
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.security.context.configuration;

import java.util.logging.Logger;
import java.util.logging.Level;
import uk.ac.soton.itinnovation.gemss.utils.configuration.ConfigurationException;
import uk.ac.soton.itinnovation.gemss.utils.configuration.Configuration;

/**
 * SecurityContextConfiguration provides configuration information for the security context component
 */
public class SecurityContextConfiguration {

    public static final String SECURITY_TOKEN_PROVIDER_FILE_NAME = "security_token_providers.xml";
    public static final String SECURITY_TRUST_PROVIDER_FILE_NAME = "security_trust_providers.xml";
    public static final String SECURITY_STORE_PROVIDER_FILE_NAME = "security_store_providers.xml";
    private static final String CONFIG_FILE_NAME = "gemss.security.context.properties";
    private static String mConfigDir = null;
    private static String fileSeparator = null;
    private static Logger mLogger = Logger.getLogger("uk.ac.soton.itinnovation.gemss.utils.configuration.Configuration");
    private Configuration mConfiguration;

    /**
     * Constructor takes nothing and assumes that the configuration directory is the current directory
     * @throws ConfigurationException
     */
    public SecurityContextConfiguration() throws ConfigurationException {
        mConfiguration = new Configuration(CONFIG_FILE_NAME);
        fileSeparator = System.getProperty("file.separator");
    }

    /**
     * Constructor requires that the configuration directory where additional configuration
     * files are stored is supplied.
     * @param configFileDir directory where additional configuration files are stored
     * @throws ConfigurationException
     */
    public SecurityContextConfiguration(String configFileDir) throws ConfigurationException {
        if(configFileDir==null) {
            throw new ConfigurationException("Security Context Configuration file invalid");
        }
        fileSeparator = System.getProperty("file.separator");
        if(configFileDir.equals(""))
            mConfiguration = new Configuration(CONFIG_FILE_NAME,configFileDir);
        else
            mConfiguration = new Configuration(configFileDir + fileSeparator + CONFIG_FILE_NAME,configFileDir);
    }

    /**
     * Retrieve the configuration directory for the security module
     * @return configuration directory
     */
    public String getConfigDirectory() {
        return mConfiguration.getConfigurationFileDir();
    }

    /**
     * Retrieve the security provider configuration object, it enables a security provider
     * to obtain further configuration information
     * @return service provider configuration
     */
    public SecurityProviderConfiguration getSecurityProviderConfiguration() {
        return new SecurityProviderConfiguration(this);
    }

    /**
     * Retrieve the configuration value for the passed configuration option name
     * @param name name of option to retrieve
     * @return configuration option value
     * @throws ConfigurationException
     */
    public String getConfigurationValue(String name) throws ConfigurationException {
        return mConfiguration.getConfigurationValue(name);
    }

    /**
     * Retrieve the path to the security token provider list
     * @return list file path
     * @throws ConfigurationException
     */
    public String getSecurityTokenProviderFilePath() throws ConfigurationException {
        String configDir = mConfiguration.getConfigurationFileDir();
        if(configDir.equals(""))
            return this.SECURITY_TOKEN_PROVIDER_FILE_NAME;
        else
            return configDir + fileSeparator + this.SECURITY_TOKEN_PROVIDER_FILE_NAME;
    }

    /**
     * Retrieve the path to the security trust provider list
     * @return list file path
     * @throws ConfigurationException
     */
    public String getSecurityTrustProviderFilePath() throws ConfigurationException {
        String configDir = mConfiguration.getConfigurationFileDir();
        if(configDir.equals(""))
            return this.SECURITY_TRUST_PROVIDER_FILE_NAME;
        else
            return configDir + fileSeparator + this.SECURITY_TRUST_PROVIDER_FILE_NAME;
    }

    public String getSecurityStoreProviderFilePath() throws ConfigurationException {
        String configDir = mConfiguration.getConfigurationFileDir();
        if(configDir.equals(""))
            return this.SECURITY_STORE_PROVIDER_FILE_NAME;
        else
            return configDir + fileSeparator + this.SECURITY_STORE_PROVIDER_FILE_NAME;
    }

    /**
     * Add a configuration option to the active configuration, no overwriting is allowed
     * @param name configuration option name
     * @param value configuration option value
     * @throws ConfigurationException
     */
    public void addConfigurationSetting(String name, String value) throws ConfigurationException {
        mConfiguration.addConfigurationSetting(name,value);
    }
}