/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Stuart E. Middleton
//      Created date:           2004/06/08
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.server;

import java.rmi.RemoteException;
import at.ac.univie.iss.service.qos.QoSHandler;

/**
 * Server side negotiation interface for the QoS negotiation handler. The QoS handler will provide all the methods
 * needed to support the Negotiation interface. The ApplicationExecuter that implements the Negotiation interface
 * is expected to convert byte64 encoded strings into actual strings before using the methods within this handlers
 * interface.
 */
public interface NegotiationHandler {

      /**
       * initialize the handler
       * @param strConfig negotiation handler configuration file
       * @param qoshandler ISS's QoS handler object, or null if this is not available
       */
      public void init( String strConfig, QoSHandler qoshandler ) throws RemoteException;

      /**
       * getJobCID handler for the getJobCID method
       * @param strNegConvID negotiation conversation ID
       * @return job conversation ID (or "" if job ID not available in this conversation)
       */
      public String getJobCID( String strNegConvID ) throws RemoteException;

      /**
       * getNegCID handler for the getJobCID method
       * @param strAccountConvID negotiation conversation ID
       * @return negoitation conversation ID (or "" if neg ID not available in this conversation)
       */
      public String getNegCID( String strAccountConvID ) throws RemoteException;

      /**
       * remove conversation from the state repository permanently. This should only be called when an auction has been
       * timed out or the acknowledge results operation has been called.
       * TODO This needs more work for final release, with overall timeouts etc.
       * @param strNegConvID negotiation conversation ID
       */
      public void deleteConversation( String strNegConvID ) throws RemoteException;

      /**
       * check to see if a conversation ID is authorized
       * for now this simply checks the conversation exists
       * TODO use the conv auth system
       * @param strConvID conversation ID
       * @return true of the conversation is authorized
       */
      public boolean queryAuthorization( String strConvID ) throws RemoteException;

      /**
       * Return true if a WSLA contract has been agreed, or false if not for whatever reason.
       * @param strConvID conversation ID
       * @return true of a contract has been agreed
       */
      public boolean queryContractAgreed( String strConvID ) throws RemoteException;

      /**
       * getWSLA handler for the getWSLA method
       * @param strNegConvID negotiation conversation ID
       * @param strServiceName service name
       * @return WSLA string in its XML format
       */
      public String getWSLA( String strNegConvID, String strServiceName ) throws RemoteException;

      /**
       * uploadRequestDesc
       * @param strNegConvID negotiation conversation ID
       * @param strRequestFilename
       */
      public void uploadRequestDesc( String strNegConvID, String strRequestFilename ) throws RemoteException;

      /**
       * auctionInform
       * @param strNegConvID negotiation conversation ID
       * @param strMessage auction XML message
       */
      public void auctionInform( String strNegConvID, String strMessage ) throws RemoteException;

      /**
       * auctionCFP
       * @param strNegConvID negotiation conversation ID
       * @param strMessage auction CFP message
       */
      public void auctionCFP( String strNegConvID, String strMessage ) throws RemoteException;

      /**
       * auctionPropose propose a bid, based on a temporary reservation with the QoS module/scheduler,
       * that has a value greater than the min bid threshold value in the last cfp.
       * when asking for a new reservation from the QoS management module the last reservation will be released. It is
       * therefore possible that if no new reservation can be made (e.g. client threshold too high), the old reservation
       * will also no longer be available.
       * If this happens thios method will return a no bid, and the ast bid will become invalid. The client will find this
       * out when the auctionRequest method is called and the service provider fails to accept the last bid.
       * @param strNegConvID negotiation conversation ID
       * @return auction propose message, with body of the WSLA proposal
       */
      public String auctionPropose( String strNegConvID ) throws RemoteException;

      /**
       * auctionAcceptProposal record the plain text WSLA that the client accepts. The next message should be a
       * request to exchange contracts, which will be compared to this accepted WSLA. There is no return since a bid made
       * by a service provider is assumed to be acceptable to that service provider. Nothing is agreed until the request
       * message is successfully processed however.
       * @param strNegConvID negotiation conversation ID
       * @param strMessage auction accept proposal message
       */
      public void auctionAcceptProposal( String strNegConvID, String strMessage ) throws RemoteException;

      /**
       * auctionRejectProposal will tell QoSManagement module to release the last WSLA reservation and will release
       * any state associated with this neg conversation. The conversation is dead as far as the service provider is
       * concerned as soon as it received this message.
       * @param strNegConvID negotiation conversation ID
       * @param strMessage auction accept proposal message
       */
      public void auctionRejectProposal( String strNegConvID, String strMessage ) throws RemoteException;

      /**
       * auctionRequest takes a input a signed contract by the client and returns a signed contract by the
       * service provider. The contract contents can be checked with the accepted plain text contract sent in an earlier
       * transaction.
       * @param strNegConvID negotiation conversation ID
       * @param strMessage auction request message, with body of a signed contract
       * @return auction inform message, with body of a signed contract
       */
      public String auctionRequest( String strNegConvID, String strMessage ) throws RemoteException;

} // end of NegotiationHandler

