/////////////////////////////////////////////////////////////////////////
//
//  University of Southampton IT Innovation Centre, 2004
//
// Copyright in this library belongs to the IT Innovation Centre of
// 2 Venture Road, Chilworth Science Park, Southampton, SO16 7NP, UK.
//
// This software may not be used, sold, licensed, transferred, copied
// or reproduced in whole or in part in any manner or form or in or
// on any media by any person other than in accordance with the terms
// of the Licence Agreement supplied with the software, or otherwise
// without the prior written consent of the copyright owners.
//
// This software is distributed WITHOUT ANY WARRANTY, without even the
// implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE, except where stated in the Licence Agreement supplied with
// the software.
//
//      Created by:             Darren Marvin
//      Created date:           2004/04/30
//      Created for project:    GEMSS
//
/////////////////////////////////////////////////////////////////////////
//
//      Dependencies: None
//
/////////////////////////////////////////////////////////////////////////
//
//      Last commit info:       $Author: $
//                              $Date: $
//                              $Revision: $
//
/////////////////////////////////////////////////////////////////////////

package uk.ac.soton.itinnovation.gemss.transportmessaging.configuration;

import java.util.logging.Logger;
import java.util.logging.Level;
import uk.ac.soton.itinnovation.gemss.utils.configuration.ConfigurationException;
import uk.ac.soton.itinnovation.gemss.utils.configuration.Configuration;


/**
 * TransportMessagingConfiguration provides access to configuration information for the
 * transport and messaging component
 */
public class TransportMessagingConfiguration {

    public static final String MESSAGE_CODEC_PROVIDER_FILE_NAME = "message_codec_providers.xml";
    public static final String MESSAGE_PROCESSOR_PROVIDER_FILE_NAME = "message_processor_providers.xml";
    public static final String PROVIDER_CLASSNAME = "gemss.transport.messaging.security.context.provider.classname";
    private static final String CONFIG_FILE_NAME = "gemss.transport.messaging.properties";
    private static final String MSG_ENFORCEMENT_FILE_NAME = "gemss.transport.messaging.encoding.policy.file";
    private static String fileSeparator = null;
    private static Logger mLogger = Logger.getLogger("uk.ac.soton.itinnovation.gemss.utils.configuration.Configuration");
    private Configuration mConfiguration;

    /**
     * Constructor takes nothing and assumes that the configuration directory is the current directory
     * @throws ConfigurationException
     */
    public TransportMessagingConfiguration() throws ConfigurationException {

      mConfiguration = new Configuration(CONFIG_FILE_NAME);
        fileSeparator = System.getProperty("file.separator");
        mLogger.log(Level.INFO,"Using configuration directory '.' for T&M");
    }

    /**
     * Constructor requires that the configuration directory where additional configuration
     * files are stored is supplied.
     * @param configFileDir directory where additional configuration files are stored
     * @throws ConfigurationException
     */
    public TransportMessagingConfiguration(String configFileDir) throws ConfigurationException {

      fileSeparator = System.getProperty("file.separator");
        if(configFileDir==null)
            throw new ConfigurationException("Transport and Messaging configuration directory is invalid");
        if(configFileDir.equals(""))
            mConfiguration = new Configuration(CONFIG_FILE_NAME,configFileDir);
        else
            mConfiguration = new Configuration(configFileDir + fileSeparator + CONFIG_FILE_NAME,configFileDir);
        mLogger.log(Level.INFO,"Using configuration directory '" + configFileDir + "' for T&M");
    }

    /**
     * Retrieve the directory path where configuration files can be found
     * @return directory path without any trailing slash
     */
    public String getConfigurationFileDir() {
        return mConfiguration.getConfigurationFileDir();
    }

    public MessagingProviderConfiguration getMessagingProviderConfiguration() {
        return new MessagingProviderConfiguration(this);
    }

    public String getMessageEnforcementFilePath() throws ConfigurationException {
      return mConfiguration.getConfigurationValue(MSG_ENFORCEMENT_FILE_NAME);
    }

    /**
     * Retrieve the configuration value for the passed configuration option name
     * @param name name of option to retrieve
     * @return configuration option value
     * @throws ConfigurationException
     */
    public String getConfigurationValue(String name) throws ConfigurationException {
        return mConfiguration.getConfigurationValue(name);
    }

    /**
     * Retrieve the path to the message codec provider list
     * @return list file path
     * @throws ConfigurationException
     */
    public String getMessageCoDecProviderFilePath() throws ConfigurationException {
        String configDir = mConfiguration.getConfigurationFileDir();
        if(configDir.equals(""))
            return this.MESSAGE_CODEC_PROVIDER_FILE_NAME;
        else
            return mConfiguration.getConfigurationFileDir() + fileSeparator + this.MESSAGE_CODEC_PROVIDER_FILE_NAME;
    }

    /**
     * Retrieve the path to the message processor provider list
     * @return list file path
     * @throws ConfigurationException
     */
    public String getMessageProcessorProviderFilePath() throws ConfigurationException {
        String configDir = mConfiguration.getConfigurationFileDir();
        if(configDir.equals(""))
            return this.MESSAGE_PROCESSOR_PROVIDER_FILE_NAME;
        else
            return mConfiguration.getConfigurationFileDir() + fileSeparator + this.MESSAGE_PROCESSOR_PROVIDER_FILE_NAME;
    }

    public String getSecurityContextProviderClassName() throws ConfigurationException {
        return mConfiguration.getConfigurationValue(PROVIDER_CLASSNAME);
    }

    /**
     * Add a configuration option to the active configuration, no overwriting is allowed
     * @param name configuration option name
     * @param value configuration option value
     * @throws ConfigurationException
     */
    public void addConfigurationSetting(String name, String value) throws ConfigurationException {
        mConfiguration.addConfigurationSetting(name,value);
    }
}